/*
 * Copyright (c) 2017 by IT Modernisation Services GmbH
 * Hamburg, Germany.  All rights reserved.
 */
package com.itms.demo.generali;

import com.itms.demo.generali.model.AvlNode;
import com.itms.demo.generali.model.ElementType;
import org.apache.log4j.Logger;
import org.springframework.stereotype.Component;

import static java.lang.Integer.max;

/**
 * @author IT Modernisation Services GmbH
 *
 *          Entity       AVLUtils
 *          Transformer  IT Modernisation Services GmbH
 *                       Hamburg, Germany
 *          On           17.06.2017
 *          Using        njema Version 4.4
 */
@Component
public class AVLUtils {
    private Logger log = Logger.getLogger(AVLUtils.class);

    /*-------------------------------------------------------------------
     * AVL-Baum für Elemente
     * Enthaltene Funktionen:
     * ======================
     * Delete(ElementToDelete,Node)            (noch nicht implement.)
     * DoubleRotateWithLeft(K3)
     * DoubleRotateWithRight(K1)
     * Find(ElementToFind,Node)
     * FindMin(Node)
     * FindMax(Node)
     * Height(Node)
     * Insert(CurrentElement,Node)
     * MakeEmpty(Node)
     * MakeEmptyGP(Node)
     * PrintTree(Node)                         (für Debugzwecke)
     * Retrieve(Node)
     * SingleRotateWithLeft(K2)
     * SingleRotateWithRight(K1)
     *
     *-------------------------------------------------------------------
     * Erklärung:
     * Im aufrufenden Programm muss der Element-Typ (der zu speichernde
     * Record) mit einem eindeutigen Key-Feld definiert werden.
     * Das Key-Feld muss logisch vergleichbar sein.
     * Die zusätzlichen Daten werden in den Data-Teil der Struktur gesp.
     * Bsp:
     * ====
     * define structure
     *   1 ElementType
     *    ,3 Key char(32000) var
     *    ,3 Data
     *      ,5 SV_NR       CHAR(04)
     *      ,5 GEB_DATUM   CHAR(10)
     *      ,5 VNAME       CHAR(50)
     *      ,5 NNAME       CHAR(50)
     * ;
     *
     * Da bisher keine Möglichkeit gefunden war, das Key-Feld auf den
     * Datenteil direkt zu verweisen so wäre hier z.B. SV_NR und
     * GEB_DATUM ein eindeutiger Schlüssel.
     *
     *
     * Folgende Schritte müssen berücksichtigt werden:
     * -) Definition von Elementtyp und Knotenstruktur (siehe oben)
     *
     *    => define structure
     *    =>   1 ElementType
     *    => ...
     *
     * Die Knotenstruktur AvlNode befindet sich hier im Include.
     *
     * -) Definition des Root-Knotens und initialisieren mit null()
     *    da sonst auf irgend einen Speicherbereich verwiesen wird
     *
     *    => dcl RootNode handle AvlNode init(null());
     *
     *    Ist der RootNode bereits verwendet worden, so muss man ihn
     *    löschen. Der allokierte Speicher wird dabei freigegeben.
     *
     *    => if RootNode ^= null() then
     *    =>    RootNode  = MakeEmpty(RootNode);
     *
     * -) Definition einer Zugriffsvariable
     *
     *    => dcl AktElement type ElementType;
     *
     * -) Befüllen des Baums mit Key und Werten (für Insert)
     *
     *   => AktElement.Key = '081519700101';
     *   => AktElement.Data.SV_NR     = '0815';
     *   => AktElement.Data.GEB_DATUM = '19700101';
     *   => AktElement.Data.VNAME     = 'Max';
     *   => AktElement.Data.NNAME     = 'Mustermann';
     *
     *   Und der eigentliche Insert, immer mit Angabe des Root-Nodes
     *
     *   => RootNode = Insert(AktElement,RootNode);
     *
     *   Der Baum wird automatisch ausbalanciert.
     *
     *
     * -) Bei Bedarf Löschen eines Elements
     *
     *   => AktElement.Key = '081519700101';
     *
     *   Und der eigentliche Delete, immer mit Angabe des Root-Nodes
     *
     *   => RootNode = Delete(AktElement,RootNode);
     *
     *   Der Baum wird automatisch ausbalanciert.
     *
     * -) Bei Bedarf Finden und "Holen" eines Elements über Hilfsknoten
     *    und RootNode
     *
     *    => dcl Position    handle AVLNode init(null());
     *    => dcl SuchElement type ElementType;
     *    =>
     *    => // Suchbegriff belegen
     *    => SuchElement.Key = '081519700101';
     *    => Position = Find(SuchElement,RootNode);
     *    => if Position ^= null() then
     *    => do;
     *    =>    SuchElement = Retrieve(Position);
     *    =>    ...
     *    => end;
     *-------------------------------------------------------------------*/

    /*-------------------------------------------------------------------
     * Delete: Löschen eines bestimmten Elements im Baum
     *-------------------------------------------------------------------
     * PARAMETER       : ElementToDelete ... zu loeschendes Element
     * handle auf Node ... Startknoten (Root)
     *-------------------------------------------------------------------
     * RETURNS         : handle auf Node ... neues Root-Element
     *-------------------------------------------------------------------*/
    @SuppressWarnings("unused")
    public AvlNode delete (ElementType elementType, AvlNode node) {
        log.error("Delete ist noch nicht implementiert !!!");
        log.error("Folgender Key wurde versucht zu löschen: " + node.getKey());
        return node;
    }

    /*-------------------------------------------------------------------
     * DoubleRotateWithLeft: doppelte Linksrotation
     * -) kann nur aufgerufen werden wenn K3 einen linken Child-knoten
     *    hat und K3's linker Childknoten einen rechtes Child-Knoten hat
     * -) führt eine links-rechts Doppelrotation aus
     * -) aktualisiert die Height, und liefert den neuen
     *    Root-Knoten retoure
     *-------------------------------------------------------------------
     * PARAMETER       : handle auf Node ... Knoten ab dem gestartet wird
     *-------------------------------------------------------------------
     * RETURNS         : handle ... handle auf Element vom Typ AvlNode
     *-------------------------------------------------------------------*/
    public AvlNode doubleRotateWithLeft(AvlNode k3) {
        // Rotiere zwischen K1 und K2
        k3.setLeft(singleRotateWithRight(k3.getLeft()));

        // Rotiere zwischen K3 und K2
        return singleRotateWithLeft(k3);
    }

    /*-------------------------------------------------------------------
     * DoubleRotateWithRight: doppelte Linksrotation
     * -) kann nur aufgerufen werden wenn K1 einen rechten Child-knoten
     *    hat und K1's rechter Childknoten einen linken Child-Knoten hat
     * -) führt eine rechts-links Doppelrotation aus
     * -) aktualisiert die Height, und liefert den neuen
     *    Root-Knoten retoure
     *-------------------------------------------------------------------
     * PARAMETER       : handle auf Node ... Knoten ab dem gestartet wird
     *-------------------------------------------------------------------
     * RETURNS         : handle ... handle auf Element vom Typ AvlNode
     *-------------------------------------------------------------------*/
    public AvlNode doubleRotateWithRight(AvlNode k1) {
        // Rotiere zwischen K3 und K2
        k1.setRight(singleRotateWithLeft(k1.getRight()));

        // Rotiere zwischen K1 und K2
        return singleRotateWithRight(k1);
    }

    /*-------------------------------------------------------------------
     * Find: Finden eines Elements im AVL-Baum
     *       (KEY-Feld ist relevant
     *-------------------------------------------------------------------
     * PARAMETER       : ElementToFind   ... gesuchtes Element (KEY !!!)
     *                   handle auf Node ... Knoten ab dem gestartet wird
     *                                       also ROOT
     *-------------------------------------------------------------------
     * RETURNS         : handle ... handle auf Element vom Typ ElementTyp
     *-------------------------------------------------------------------*/
    public AvlNode find(ElementType elementToFind, AvlNode node){
        // Wenn der Startknoten null() ist gibt es nicht zu finden
        if (node == null) {
            // kommt einem return(null()) gleich
            return node;
        } else {
            // Vergleichskrit. Key ist < als Element des akt. Knoten
            if (elementToFind.getKey() < node.getElement().getKey()) {
                // auf der linken Seite im Baum weitersuchen
                return find(elementToFind, node.getLeft());
            } else {
                // Vergleichskrit. Key ist > als Element des akt. Knoten
                if (elementToFind.getKey() > node.getElement().getKey()) {
                    // auf der rechten Seite im Baum weitersuchen
                    return find(elementToFind, node.getRight());
                } else {
                    // gesuchtes Element ist in aktuellem Knoten
                    return node;
                }
            }
        }
    }

    /*-------------------------------------------------------------------
     * FindMin: Finden des kleinsten Elements im AVL-Baum
     *-------------------------------------------------------------------
     * PARAMETER       : handle auf Node ... Knoten ab dem gestartet wird
     *                                       also ROOT
     *-------------------------------------------------------------------
     * RETURNS         : handle ... handle auf Element vom Typ AvlNode7
     *-------------------------------------------------------------------*/
    public AvlNode findMin(AvlNode node) {
        // wenn Knoten nicht null()
        if (node != null) {
            // solange auf linker Seite ein Element vorhanden
            do {
                // Neuer Ausgangspunkt = linker Knoten
                node = node.getLeft();
            } while (node.getLeft() != null); // do
        } // Node ^= null()

        return node;
    }

    /*-------------------------------------------------------------------
     * FindMax: Finden des groessten Elements im AVL-Baum
     *-------------------------------------------------------------------
     * PARAMETER       : handle auf Node ... Knoten ab dem gestartet wird
     *                                       also ROOT
     *-------------------------------------------------------------------
     * RETURNS         : handle ... handle auf Element vom Typ AvlNode
     *-------------------------------------------------------------------*/
    public AvlNode findMax(AvlNode node) {
        // wenn Knoten nicht null()
        if (node != null) {
            // solange auf rechter Seite ein Element vorhanden
            do {
                // Neuer Ausgangspunkt = rechter Knoten
                node = node.getRight();
            } while (node.getRight() != null); // do

        } // Node ^= null()

        return node;
    }

    /*-------------------------------------------------------------------
     * Height: Ermitteln der Height des AVL-Knotens
     *         Retourniert die Hoehe des Knoten, oder -1, wenn null()
     *-------------------------------------------------------------------
     * PARAMETER       : handle auf Node ... Knoten ab dem gestartet wird
     *-------------------------------------------------------------------
     * RETURNS         : handle ... handle auf Element vom Typ AvlNode
     *-------------------------------------------------------------------*/
    public Integer height(AvlNode node) {
        Integer retVal;

        // Handelt es sich um null() ==> -1
        if (node == null) {
            retVal = -1;
        } else {
            retVal = node.getHeight();
        }

        return retVal;
    }


    /*-------------------------------------------------------------------
     * Insert: Einfuegen eines neuen Elements in den AVL-Baum
     *-------------------------------------------------------------------
     * PARAMETER       : handle auf Node ... Knoten ab dem gestartet wird
     *                                       also ROOT
     *-------------------------------------------------------------------
     * RETURNS         : handle ... handle auf Element vom Typ AvlNode
     *-------------------------------------------------------------------*/
    public AvlNode insert(ElementType currentElement, AvlNode node) {
        Integer heightLeft;
        Integer heightRight;

        // ist der Baum leer ?
        if (node == null) {
            // Initialisieren eines handle auf einen AvlNode
            node = new AvlNode();

            // Speichern des uebergebenen Elements und initialisieren der
            // Child-Knoten sowie der Height für erstes Element im Baum
            node.setElement(currentElement);
            node.setHeight(0);
            node.setBalance(0);
            node.setLeft(null);
            node.setRight(null);
        } else { // Node = null()
            // aktuelles Element kleiner dem angegebenen Knoten, gehoert
            // also irgendwo links vom aktuellen Knoten eingefuegt
            if (currentElement.getKey() < node.getElement().getKey()) {
                // rekursiv versuchen an linkes Child-Element einzufuegen
                node.setLeft(insert(currentElement, node.getLeft()));
                // Wenn Height nicht stimmt => rotieren
                if (height(node.getLeft()) - height(node.getRight()) == 2) {
                    // wenn neuer Key < als Element von linken Node */
                    if (currentElement.getKey() < node.getLeft().getElement().getKey()) {
                        // einfaches Linksrotieren
                        node = singleRotateWithLeft(node);
                    } else {
                        // doppeltes Linksrotieren
                        node = doubleRotateWithLeft(node);
                    }
                } // Height nicht OK
            } else { // CurrentElement.Key < Node=>Element.Key
                // wenn neuer Key > als Element von linken Node
                if (currentElement.getKey() > node.getElement().getKey()) {
                    // rekursiv versuchen an rechtes Child-Element einzufuegen
                    node.setRight(insert(currentElement, node.getRight()));
                    // Wenn Height nicht stimmt => rotieren
                    if (height(node.getRight()) - height(node.getLeft()) == 2) {
                        if (currentElement.getKey() > node.getRight().getElement().getKey()) {
                            // einfaches Rechtsrotieren
                            node = singleRotateWithRight(node);
                        } else {
                            // doppeltes Rechtsrotieren
                            node = doubleRotateWithRight(node);
                        }
                    } // Height nicht OK
                } // CurrentElement.Key > Node=>Element.Key
            } // Knotenelement bereits im Baum, nichts machen
        } // else - Node = null()

        /* Height und Balance neu berechnen */
        heightLeft    = height(node.getLeft());
        heightRight   = height(node.getRight());
        node.setHeight(max(heightLeft, heightRight) + 1);
        node.setBalance(heightRight - heightLeft);

        // originale Version zur Dokumentation erhalten, Balance nicht ber.
        // Node=>Height = Max(Height(Node=>Left), Height(Node=>Right)) + 1;

        return node;
    }

    /*-------------------------------------------------------------------
     *                   MakeEmpty: Leeren eines AVL-Baums
     *-------------------------------------------------------------------
     * PARAMETER       : handle auf Node ... Startknoten
     *-------------------------------------------------------------------
     * RETURNS         : null() ... immer null()
     *-------------------------------------------------------------------*/
    @SuppressWarnings("unused")
    public AvlNode makeEmpty(AvlNode node) {
        // return null-value because all elements are freed to re-init
        // root-node with null()
        return null;
    }

    /*-------------------------------------------------------------------
     * MAKEEMPTYGP: LEEREN DES AVL-BAUMES
     *-------------------------------------------------------------------
     * PARAMETER       : HANDLE AUF NODE ... STARTKNOTEN
     *-------------------------------------------------------------------
     * RETURNS         : NULL() ... IMMER NULL()
     *-------------------------------------------------------------------*/
    @SuppressWarnings("unused")
    public AvlNode makeEmptyGP(AvlNode node) {
        // return null-value because all elements are freed to re-init */
        // root-node with null() */
        return null;
    }

    /*-------------------------------------------------------------------
     * PrintTree: rekursive Ausgabe des KEY-Feldes des gesamten Baums
     *            inklusive der Hoehe und Balance sowie das Key-Feld des
     *            jeweils linken und rechten Knotens
     *-------------------------------------------------------------------
     * PARAMETER       : handle auf Node ... Startknoten
     *-------------------------------------------------------------------
     * RETURNS         : ---
     *-------------------------------------------------------------------*/
    public void printTree(AvlNode node) {
        if (node != null) {
            // rekursive Ausgaben des linken Teilbaums
            printTree(node.getLeft());

            // Ausgabe der Informationen in Joblog
            log.info("--------------------------------------------------------------------------------");
            log.info("Key       : " + node.getElement().getKey());
            log.info("Height    : " + node.getHeight());
            log.info("Balance   : " + node.getBalance());
            // gibt es einen linken Knoten, dann Ausgabe dessen Key's
            if (node.getLeft() != null) {
                log.info("Left Node Key : " + node.getLeft().getElement().getKey());
            }
            // gibt es einen rechten Knoten, dann Ausgabe dessen Key's
            if (node.getRight() != null) {
                log.info("Right Node Key: " + node.getRight().getElement().getKey());
            }
            log.info("--------------------------------------------------------------------------------");

            // rekursive Ausgaben des rechten Teilbaums
            printTree(node.getRight());
        }
    }

    /*-------------------------------------------------------------------
     * Retrieve: Lesen eines Elements in angegebenem Knoten
     *-------------------------------------------------------------------
     * PARAMETER       : handle auf Node ... zu bearbeitender Knoten
     *-------------------------------------------------------------------
     * RETURNS         : ElementType     ... angefordertes Element
     *-------------------------------------------------------------------*/
    public ElementType retrieve(AvlNode node) {
        // angefordertes Element retournieren
        return node.getElement();
    }

    /*-------------------------------------------------------------------
     * SingleRotateWithLeft: einfache Linksrotation
     * -) kann nur aufgerufen werden wenn K2 einen linken Knoten hat
     * -) Rotiert die Knoten K2 mit seinem linken Child-Knoten,
     *    aktualisiert die Height und liefert den neuen
     *    Root-Knoten retoure
     *-------------------------------------------------------------------
     * PARAMETER       : handle auf Node ... Knoten ab dem gestartet wird
     *-------------------------------------------------------------------
     * RETURNS         : handle ... handle auf Element vom Typ AvlNode
     *-------------------------------------------------------------------*/
    public AvlNode singleRotateWithLeft(AvlNode k2) {
        AvlNode k1;
        Integer heightLeft;
        Integer heightRight;

        // tausche Knoten über Hilfsknoten K1 aus
        k1 = k2.getLeft();
        k2.setLeft(k1.getRight());
        k1.setRight(k2);

        // neue Height und Balance ermitteln und updaten
        heightLeft  = height(k2.getLeft());
        heightRight = height(k2.getRight());
        k2.setHeight(max(heightLeft, heightRight) + 1);
        k2.setBalance(heightRight - heightLeft);

        heightLeft  = height(k1.getLeft());
        heightRight = k2.getHeight();
        k1.setHeight(max(heightLeft, heightRight) + 1);
        k1.setBalance(heightRight - heightLeft);

        // originale Berechnungsformel aufheben, Balance nicht beruecks.
        // K2=>Height = Max( Height( K2=>Left ), Height( K2=>Right ) ) +1;
        // K1=>Height = Max( Height( K1=>Left ), K2=>Height ) + 1;

        return k1;
    }

    /*-------------------------------------------------------------------
     * SingleRotateWithRight: einfache Rechtsrotation
     * -) kann nur aufgerufen werden wenn K1 einen rechten Knoten hat
     * -) Rotiert die Knoten K1 mit seinem rechten Child-Knoten
     *    aktualisiert die Height, und liefert den neuen
     *    Root-Knoten retour
     *-------------------------------------------------------------------
     * PARAMETER       : handle auf Node ... Knoten ab dem gestartet wird
     *-------------------------------------------------------------------
     * RETURNS         : handle ... handle auf Element vom Typ AvlNode
     *-------------------------------------------------------------------*/
    public AvlNode singleRotateWithRight(AvlNode k1) {
        AvlNode k2;
        Integer heightLeft;
        Integer heightRight;

        // tausche Knoten über Hilfsknoten K2 aus
        k2 = k1.getRight();
        k1.setRight(k2.getLeft());
        k2.setLeft(k1);

        // neue Height ermitteln und updaten
        heightLeft  = height(k1.getLeft());
        heightRight = height(k1.getRight());
        k1.setHeight(max(heightLeft, heightRight) + 1);
        k1.setBalance(heightRight - heightLeft);

        heightLeft  = height(k2.getRight());
        heightRight = k1.getHeight();
        k2.setHeight(max(heightLeft, heightRight) + 1);
        k2.setBalance(heightRight - heightLeft);

        // originale Berechnungsformel aufheben, Balance nicht beruecks.
        // K1=>Height = Max( Height( K1=>Left ), Height( K1=>Right ) ) + 1;
        // K2=>Height = Max( Height( K2=>Right ), K1=>Height ) + 1;

        return k2;
    }
}
