/*
 * Copyright (c) 2017 by IT Modernisation Services GmbH
 * Hamburg, Germany.  All rights reserved.
 */
package com.itms.demo.oews;

import com.itms.demo.oews.common.model.ExecuteSqlResult;
import com.itms.demo.oews.common.model.SqlCa;
import com.itms.demo.oews.common.service.BuiltinFunctions;
import com.itms.demo.oews.common.service.DBFunctions;
import com.itms.demo.oews.common.service.FileFunctions;
import com.itms.demo.oews.model.Db2Planfld;
import com.itms.demo.oews.model.Unaligned;
import com.itms.demo.oews.service.Alahpi0;
import com.itms.demo.oews.service.Aldbhi0;
import com.itms.demo.oews.service.Alerri0;
import com.itms.demo.oews.service.Fihipi0;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Component;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;

/**
 * @author IT Modernisation Services GmbH
 *
 *          Entity       FICCPA0
 *          Transformer  IT Modernisation Services GmbH
 *                       Hamburg, Germany
 *          On           05.10.2017
 *          Using        njema Version 4.4
 */
@Component
public class FICCPA0 {
    private Logger log = Logger.getLogger(FICCPA0.class);

    @Autowired
    private DBFunctions DBFunctions;

    @Autowired
    private BuiltinFunctions builtinFunctions;

    @Autowired
    private FileFunctions fileFunctions;

    @Autowired
    private Alahpi0 alahpi0;

    @Autowired
    private Aldbhi0 aldbhi0;

    @Autowired
    private Fihipi0 fihipi0;

    @Autowired
    private Alerri0 alerri0;

    /* *******************************************************************
     *
     * FICPPA0: Nicht-CPP Faehigkeit einspielen
     *
     * AUTOR Gottfried Huber   / ÖWS         DATUM 2017.07.24
     *
     *********************************************************************
     * PROGRAMMTYP PL/I, BATCH, MIT DB2
     *
     *********************************************************************
     * BESCHREIBUNG:
     *
     * Verwendete sequentielle Datenbestände:
     *
     * DB2PLAN:
     *    SSID        CHAR(4)
     *    BLANK       CHAR(1)
     *    DB2QUAL     CHAR(8)
     *
     * IN:
     *    Output aus Compare zw. altem und aktuellem Nicht-CCP Bestand
     * NEWFILE:
     *    Aktueller CCP-File
     * OWSOUT:
     *    File fuer Mail an PF-PSS und PF-Datenservice
     *
     * ******************************************************************
     * AENDERUNGSHISTORIE:
     * ===================
     * 20170724 HG NEUERSTELLUNG  17-0016
     *
     * ******************************************************************
     *
     * S T A N D A R D - D E K L A R A T I O N E N
     *
     * ******************************************************************/

    /* -------------------------------------------------------------------
     * Globale Programmstruktur
     * -------------------------------------------------------------------*/
    private Unaligned unaligned = new Unaligned();

    /* -------------------------------------------------------------------
     * CURSORDEKLARATIONEN
     * -------------------------------------------------------------------*/

    /* *******************************************************************
     *
     *                 U S E R - D E K L A R A T I O N E N
     *
     * *******************************************************************
     * -------------------------------------------------------------------
     * PROGRAMMEIGENE COMMAREA/PARAMETER
     * -------------------------------------------------------------------*/
    private String parms;

    /* -------------------------------------------------------------------
     * BENUTZERDEFINIERTE KONSTANTEN
     * -------------------------------------------------------------------*/
    private static final String INSERT = "I";
    private static final String DELETE = "D";

    private CursStruct cursStruct;

    /* -------------------------------------------------------------------
     * DIVERSE VARIABLENDEFINITIONEN
     * -------------------------------------------------------------------*/
    private Db2Planfld db2Planfld = new Db2Planfld();

    // generated identifier for the file handling
    private String identifier = fileFunctions.getIdentifier();

    /*   **DB2-ANFANG
     *   *PROJEKT=OEWSFVSD
     *   **DB2-ENDE
     */
    public void execute(String parms) {
        try {
            this.parms = parms;
            /* *******************************************************************
             *
             *                     V E R A R B E I T U N G
             *
             * *******************************************************************
             * *******************************************************************
             *                         ANFANGSROUTINE
             * *******************************************************************/

            /* ANFANGSMELDUNG INS LOG SCHREIBEN                                  */
            log.info("Programm:" + unaligned.getProgram().getName() +
                    " START: " + builtinFunctions.getDateTime() +
                    " Parameter: " + parms);

            anfrInit();

            /* NEUEN CCP-FILE AUF VOLLSTAENDIGKEIT/KORREKTHEIT PRUEFEN           */
            if (!anfrCheckCcpFile()) {
                /* ANM.: ENTSPRECHENDE FEHLERMELDUNGEN WERDEN IN DER FKT. AUSGEG. */
                /*       KEINE WEITERE VERARBEITUNG MEHR DURCHFUEHREN             */
                return; /* ----------------------------------------------------> */
            } /* IF */

            /* ********************************************************************
             *                       HAUPTVERARBEITUNG
             * ********************************************************************/
            /* INPUTFILE VERARBEITEN                                          */
            veraVerarbeiteFile();

            /* *******************************************************************
             *                         ENDROUTINE
             * *******************************************************************/
            endrVera();

            /* ENDEMELDUNG INS LOG SCHREIBEN                                     */
            log.info("Programm:" + unaligned.getProgram().getName() +
                    " ENDE: " + builtinFunctions.getDateTime());
        } catch (Exception e) {
            alerri0.onError(e, unaligned);
        } finally {
            fileFunctions.closeFile("DB2PLAN", identifier);
            fileFunctions.closeFile("IN", identifier);
            fileFunctions.closeFile("NEWFILE", identifier);
            fileFunctions.closeFile("OWSOUT", identifier);
        }
    }

    /* *******************************************************************
     *             UNTERPROGRAMME
     * *******************************************************************
     * -------------------------------------------------------------------
     *                   ANFR_INIT: Initialisierungen
     * -------------------------------------------------------------------
     * PARAMETER       : ---
     * -------------------------------------------------------------------
     * GLOBALE VARIABLE: DB2PLANFLD
     *                   GLOBAL
     *                   IN
     *                   NEWFILE
     *                   PARMS
     * -------------------------------------------------------------------
     * RETURNS         : ---
     * -------------------------------------------------------------------*/
    private void anfrInit() throws Exception {
        String fm;

        /* DB2-Infos aus Vorlaufkarte lesen und Verbindung aufbauen       */
        fileFunctions.openFile("DB2PLAN", identifier);
        db2Planfld.setAll(fileFunctions.readLineFromFile("DB2PLAN", identifier));
        fileFunctions.closeFile("DB2PLAN", identifier);

        /* Verbindung zur Datenbank aufbauen */
        fm = aldbhi0.db2Open(db2Planfld.getDb2ssid(), db2Planfld.getDb2plan());
        if (!fm.isEmpty()) {
            log.error("FEHLER: KEINE VERBINDUNG ZU DB2:" + fm);
            throw new Exception();
        }

        /* DIE DEM PROGRAMM UEBERGEBENEN PARAMETER PRUEFEN (FUTURE USE)   */

        /* OEFFNEN DER FILES FUER WEITERFUEHRENDE VERARBEITUNG            */
        fileFunctions.openFile("IN", identifier);
        fileFunctions.openFile("NEWFILE", identifier);
        fileFunctions.openFile("OWSOUT", identifier);
    }

    /* -------------------------------------------------------------------
     * ANFR_CHECK_CCP_FILE: NEUEN CCP-FILE PRUEFEN
     * -------------------------------------------------------------------
     * PARAMETER       : ---
     * -------------------------------------------------------------------
     * GLOBALE VARIABLE: GLOBAL
     *                   NEWFILE
     *                   EOF_NEWFILE
     * -------------------------------------------------------------------
     * RETURNS         : BIT(1) ALIGNED
     *                   '0'B...FILE FEHLERHAFT
     *                   '1'B...FILE KORREKT
     * -------------------------------------------------------------------*/
    private boolean anfrCheckCcpFile() throws Exception {
        String inStr;
        Integer anzWp = 0;
        String num8C = "";

        /* DATENSATZ EINLESEN                                             */
        inStr = fileFunctions.readLineFromFile("NEWFILE", identifier);

        /* IN DER ERSTEN ZEILE BEFINDET SICH DAS DATUM DER GENERIERUNG    */
        /* DIESES MUSS MIT DEM LAUFDATUM UEBEREINSTIMMEN                  */
        /* DATUMSFORMAT IM FILE JJJJMMTT                                  */
        if (!builtinFunctions.getDateTime().substring(0, 8).equals(inStr.substring(0, 8))) {
            writeMail("============================================");
            writeMail("ACHTUNG: CCP-FILE IST VERALTET/FALSCH.");
            writeMail("DATUM IM FILE:" + inStr.substring(0, 8));
            writeMail("=> UNBEDINGT DEN AKTUELLEN GDG-FILE");
            writeMail("   ERNEUT ABHOLEN (SOBALD VORHANDEN).");
            writeMail("============================================");
            return false; /* -----------------------------------------> */
        } /* IF */

        /* INFO FUER PROGRAMMDURCHFUEHRUNG                                */
        log.info("DATUM IM FILE:" + inStr.substring(0, 7));

        /* NAECHSTE ZEILE EINLESEN                                        */
        inStr = fileFunctions.readLineFromFile("NEWFILE", identifier);

        /* SOLANGE DATENSAETZE VORHANDEN SIND                             */
        while (fileFunctions.hasMore("NEWFILE", identifier)) {
            if (inStr.substring(0,1).matches("[A-Z]+")) {
                /* ISIN-ZEILE                                               */
                anzWp++;
            } else {
                /* ZIFFER ZU BEGINN => LETZTE ZEILE MIT ANZAHL DER ISINS    */
                num8C = inStr.substring(0,8).trim();
            } /*IF*/
            inStr = fileFunctions.readLineFromFile("NEWFILE", identifier);
        } /* ^EOF_IN */

        /* SUMME ISINS STIMMT NICHT MIT DER ANGEGEBENEN ANZAHL UEBEREIN   */
        /* BZW. ANZAHL DER ISINS ZU GERING (WEITERER SICHERHEITSEINBAU)   */
        /* TRIM WICHTIG */
        /* SOLLTE NICHT SEIN                           */
        if (!anzWp.equals(Integer.valueOf(num8C.trim())) && anzWp > 2) {
            writeMail("=============================================");
            writeMail("ACHTUNG: CCP-FILE IST LEER ODER FEHLERHAFT");
            writeMail("ANZAHL LT. FILE:" + num8C);
            num8C = builtinFunctions.pictureToString(anzWp, 8);
            writeMail("ANZAHL LT. PRUEFUNG:" + num8C);
            writeMail("=> UNBEDINGT DEN AKTUELLEN GDG-FILE ");
            writeMail("   ERNEUT ABHOLEN (SOBALD KORREKT).");
            writeMail("=============================================");
            return false; /* -----------------------------------------> */
        } /* IF */

        /* INFO FUER PROGRAMMDURCHFUEHRUNG                                */
        log.info("ANZAHL LT. FILE:" + inStr);

        return true;
    }

    /* -------------------------------------------------------------------
     *                   ENDR_AUSGABE_STATISTIK: ENDESTATISTIK AUSG.
     * -------------------------------------------------------------------
     * PARAMETER       : ---
     * -------------------------------------------------------------------
     * GLOBALE VARIABLE: GLOBAL
     * -------------------------------------------------------------------
     * RETURNS         : ---
     * -------------------------------------------------------------------*/
    private void endrAusgabeStatistik() {
        String num8C;

        num8C = builtinFunctions.pictureToString(unaligned.getStatistik().getAnzVerarb(), 8);
        log.info("ANZAHL VERARBEITETER DATENSAETZE: " + num8C);
        num8C = builtinFunctions.pictureToString(unaligned.getStatistik().getAnzStatusM(), 8);
        log.info("ANZAHL DATENSäTZE STATUS M      : " + num8C);
        num8C = builtinFunctions.pictureToString(unaligned.getStatistik().getAnzStatusI(), 8);
        log.info("ANZAHL DATENSäTZE STATUS I      : " + num8C);
        num8C = builtinFunctions.pictureToString(unaligned.getStatistik().getAnzStatusD(), 8);
        log.info("ANZAHL DATENSäTZE STATUS D      : " + num8C);
        num8C = builtinFunctions.pictureToString(unaligned.getStatistik().getAnzSfi(), 8);
        log.info("ANZAHL SFI-DATENSäTZE           : " + num8C);
        num8C = builtinFunctions.pictureToString(unaligned.getStatistik().getAnzError(), 8);
        log.info("ANZAHL FEHLER \\ WARNUNGEN       : " + num8C);
    }

    /* -------------------------------------------------------------------
     *                   ENDR_VERA: ENDROUTINE VERARBEITEN
     * -------------------------------------------------------------------
     * PARAMETER       : ---
     * -------------------------------------------------------------------
     * GLOBALE VARIABLE: IN
     *                   OUT
     * -------------------------------------------------------------------
     * RETURNS         : ---
     * -------------------------------------------------------------------*/
    private void endrVera() {
        /* FILES SCHLIESZEN                                               */
        fileFunctions.closeFile("IN", identifier);
        fileFunctions.closeFile("NEWFILE", identifier);

        /* DATENBANKVERBINDUNG ABBAUEN                                    */
        aldbhi0.db2Close();

        /* PROGRAMMSTATISTIK AUSGEBEN                                     */
        endrAusgabeStatistik();
    }

    /* -------------------------------------------------------------------
     * VERA_VERARBEITE_FILE: OUTPUTBESTAND AUS SUPERCOMPARE VERARBEITEN
     * -------------------------------------------------------------------
     * PARAMETER       : ---
     * -------------------------------------------------------------------
     * GLOBALE VARIABLE: GLOBAL
     *                   VERARB_KZ
     *                   IN
     *                   NEWFILE
     *                   EOF_IN
     *                   EOF_NEWFILE
     * -------------------------------------------------------------------
     * RETURNS         : ---
     * -------------------------------------------------------------------*/
    private void veraVerarbeiteFile() throws Exception {
        String inStr;
        String aktStat = "";

        /* DATENSATZ EINLESEN                                             */
        inStr = fileFunctions.readLineFromFile("IN", identifier);

        /* SOLANGE DATENSAETZE VORHANDEN SIND                             */
        while(!fileFunctions.hasMore("IN", identifier)) {
            unaligned.getStatistik().setAnzVerarb(unaligned.getStatistik().getAnzVerarb() + 1);
            if (inStr.substring(0, 1).equals("*")) {
                /* MERKEN DES AKTUELLEN STATUS                           */
                aktStat = inStr.substring(1, 2);
            } else  if (inStr.trim().length() <= 8) { /* DATUMSFELD, KONTROLLSUMME*/
                /* NICHTS ZU TUN. ZEILE WIRD UEBERLESEN                  */
            } else {
                veraInput(aktStat, inStr);
            }

            /* NAECHSTEN DS LESEN                                          */
            inStr = fileFunctions.readLineFromFile("IN", identifier);
        } /* ^EOF_IN */
    }

    /* *******************************************************************
     *                   VERA_INPUT
     * -------------------------------------------------------------------
     * PARAMETER       : AKTUELLE SATZART (IDM)      CHAR(1)
     *                 : DATENSATZ AUS FILE          CHAR(*)
     * -------------------------------------------------------------------
     * GLOBALE VARIABLE: GLOBAL
     *                   OUT
     * -------------------------------------------------------------------
     * RETURNS         : ---
     * ********************************************************************/
    private void veraInput(String viSartC, String viInC) throws Exception {
        String sfiStr;
        String isinStr;

        isinStr = viInC.substring(0, 12);

        /* ANM: STRUKTUR CURS_STRUCT WIRD IN FKT. BEFUELLT                */
        if (db2CheckIdent(isinStr) == 100) {
            /* ISIN NOCH NICHT IN FIVS VORHANDEN => HINWEISMELDUNG          */
            /* KEINE VERARBEITUNG                                           */
            alahpi0.db2ErrorIns(unaligned.getProgram().getName(), isinStr, isinStr + " NICHT IN FIVS VORHANDEN",
                    unaligned);
            return; /* ---------------------------------------------------> */
        } else{
            /* ISIN NICHT MEHR AKTIV => HINWEISMELDUNG                      */
            if (!cursStruct.instrumentenstatus.trim().contains(":EMIT:NOT:WLOS:")) {
                alahpi0.db2ErrorIns(unaligned.getProgram().getName(), isinStr, isinStr + " NICHT MEHR AKTIV",
                        unaligned);
                return; /* ------------------------------------------------> */
            } /* IF */
        } /* IF */

        /* VERARBEITUNG JE NACH "SATZART" DES COMPARES (I, D, M)          */
        switch (viSartC) {
            /* NEUER / GEAENDERTER DATENSATZ => PRUEFEN UND GGF. EINTRAGEN */
            case "I":
                if (cursStruct.iFiKatSchema == -1) {
                    /* AUCH IN DIESEM FALL MUSS GEPRUEFT WERDEN, OB BEREITS
                     * EIN FIKAT EINTRAG EXISTIERT. GRUND: AENDERT SICH DIE
                     * WP-BEZEICHNUNG IM TEXT DANN SCHEINT DIESE ZEILE ALS
                     * NEUER EINTRAG AUF.
                     * GENERIEREN D. SCHNITTSTELLENSTRUKTUR                  */
                    sfiStr = genCcpsatz(INSERT);
                    /* ... UND INSERT AUF DIE SS-TABELLE SFI                 */
                    fihipi0.db2SfiIns(sfiStr, unaligned);
                } /* IF */
                /* STATISTIK HOCHZAEHLEN                                    */
                veraStatistik(viSartC);
                break; /* I */
            /* DATENSATZ BEREITS VORHANDEN => PRUEFEN UND GGF. EINTRAGEN   */
            case "M":
                if (cursStruct.iFiKatSchema == -1) {  /*WG. OUTER JOIN */
                    /* FIKAT-EINTRAG WAR NOCH NICHT VORHANDEN DA DAS FI ERST
                     * IN FIVS ANGELEGT WORDEN IST => EINTRAGEN
                     * GENERIEREN D. SCHNITTSTELLENSTRUKTUR                  */
                    sfiStr = genCcpsatz(INSERT);
                    /* ... UND INSERT AUF DIE SS-TABELLE SFI                 */
                    fihipi0.db2SfiIns(sfiStr, unaligned);
                } /* IF */
                /* STATISTIK HOCHZAEHLEN                                    */
                veraStatistik(viSartC);
                break; /* M */
            /* DATENSATZ N. MEHR VORHANDEN => PRUEFEN UND GGF. ENTFERNEN   */
            case "D":
                if (cursStruct.iFiKatSchema == -1) { /*WG. OUTER JOIN */
                    /* FIKAT-EINTRAG WAR EV. NICHT VORHANDEN DA DAS FI ERST
                     * IN FIVS ANGELEGT WORDEN IST. JETZT IST ES NICHT MEHR
                     * AUF DER LISTE OBEN. => AUCH VOR DER LOESCHUNG PRUEFEN */
                    sfiStr = genCcpsatz(DELETE);
                    /* ... UND INSERT AUF DIE SS-TABELLE SFI                 */
                    fihipi0.db2SfiIns(sfiStr, unaligned);
                } /* IF */
                /* STATISTIK HOCHZAEHLEN                                    */
                veraStatistik(viSartC);
                break; /* D */
            default:
                log.error("UNBEKANNTE SATZART: " + viSartC);
                throw new Exception(); /* --------------------------------> */
        } /* SELECT */
    }

    /* *******************************************************************
     *                   VERA_STATISTIK
     * -------------------------------------------------------------------
     * PARAMETER       : DATENSATZART                CHAR(1)
     * -------------------------------------------------------------------
     * GLOBALE VARIABLE: GLOBAL
     * -------------------------------------------------------------------
     * RETURNS         : ---
     * ********************************************************************/
    private void veraStatistik(String vsArtC) {
        switch (vsArtC) {
            /* I...NEUER ODER GEäNDERTER DATENSATZ                         */
            case "I":
                unaligned.getStatistik().setAnzStatusI(unaligned.getStatistik().getAnzStatusI() + 1);
                break; /* I */
            /* D...GELOESCHTE DATENSAETZE                                  */
            case "D":
                unaligned.getStatistik().setAnzStatusD(unaligned.getStatistik().getAnzStatusD() + 1);
                break; /* D */
            /* M...GLEICHE DATENSAETZE                                     */
            case "M":
                unaligned.getStatistik().setAnzStatusM(unaligned.getStatistik().getAnzStatusM() + 1);
                break; /* M */
            default:
        } /* SELECT */
    }

    /* *******************************************************************
     * DIVERSE HILFSUNKTIONEN UND PROZEDUREN
     * *******************************************************************
     * *******************************************************************
     * GEN_CCPSATZ_INS : ERSTELLEN DER SS-STRUKTUR FUER NICHT CCP-KZ
     * -------------------------------------------------------------------
     * PARAMETER       : SATZ FUER LOESCHUNG BZW. EINFUEGEN    CHAR(1)
     * -------------------------------------------------------------------
     * GLOBALE VARIABLE: CURS_STRUCT
     * -------------------------------------------------------------------
     * RETURNS         : SST.-SATZ FUER SFI          CHAR(2000) VARYING
     * *******************************************************************/
    private String genCcpsatz(String gcIdC) {
        String retStr;

        retStr = fihipi0.genStruk(11620,"6/F") +
                fihipi0.genStruk(11664,"6/I;8/ISIN;7/" + cursStruct.ident + ";8/" + cursStruct.statusIdent) +
                fihipi0.genStruk(11626,"6/" + gcIdC + ";8/CCPX;8/CCPN");

        return retStr;
    }

    /* *******************************************************************
     * MELDUNG AUF FILEBESTAND FUER MAIL + SYSOUT SCHREIBEN
     * -------------------------------------------------------------------
     * PARAMETER       : MELDUNG                           CHAR(*)
     * -------------------------------------------------------------------
     * GLOBALE VARIABLE: OWSOUT                            FILE
     * -------------------------------------------------------------------
     * RETURNS         : ---
     * *******************************************************************/
    private void writeMail(String wmMeldungC) throws Exception {
        log.info(wmMeldungC);

        fileFunctions.writeToFile("OWSOUT", identifier, wmMeldungC);
    }

    /* *******************************************************************
     *             DB2_UNTERPROGRAMME
     * *******************************************************************
     * -------------------------------------------------------------------
     *             DB2_CHECK_IDENT
     * -------------------------------------------------------------------
     * PARAMETER       : ISIN                              CHAR(12)
     * -------------------------------------------------------------------
     * GLOBALE VARIABLE: DCLIDENT
     *                   DCLKFI
     *                   DCLWP
     *                   SQLCA.SQLCODE
     * -------------------------------------------------------------------
     * RETURNS         : SQLCA.SQLCODE                     BIN FIXED(31)
     * -------------------------------------------------------------------*/
    private Integer db2CheckIdent(String dciIsinStr) {
        ExecuteSqlResult result = DBFunctions.executeSql("SELECT I.FK_ID AS fiId, " +
                "SUBSTR(I.IDENT, 1, 12) AS ident, " +
                "I.STATUS AS statusIdent, " +
                "K.INSTRUMENTENSTATUS AS instrumentenstatus, " +
                "K.FI_SUB_TYP AS fiSubTyp, " +     /* FUTURE USE */
                "K.FI_SUB_ART AS fiSubArt, " +     /* FUTURE USE */
                "F.FI_KAT_SCHEMA AS fiKatSchema " +
                "FROM IDENT I, " +
                "KFI K LEFT OUTER JOIN FIKAT F " +
                "ON(K.FI_ID = F.FK_ID " +
                "AND F.FI_KAT_SCHEMA = 'CCPX' " +
                "AND F.FI_KAT_IDENT = 'CCPN' " +
                "AND F.FIKAT_TST_BIS = '2800-12-31-23.59.59.000000' " +
                ") " +
                "WHERE I.IDENT = '" + dciIsinStr + "' " +
                "AND I.SCHEMA = 'ISIN' " +
                "AND I.STATUS<> 'ABG' " +
                "AND I.IDENT_TST_BIS = '2800-12-31-23.59.59.000000' " +
                "AND I.FK_ID = K.FI_ID " +
                "FETCH FIRST 1 ROWS ONLY " +
                "WITH UR; ", cursStruct.getMapper());

        /* -------------------------------------------------------------------
         * DB2-SPECIALS
         * -------------------------------------------------------------------*/
        SqlCa sqlCa = result.getSqlCa();
        cursStruct = (CursStruct) result.getResultObject();

        switch(sqlCa.getSqlCode()) {
            case 0:
            case 100:
                break;
            default:
                aldbhi0.db2Error("DB2_CHECK_IDENT:" + dciIsinStr);
        }

        return sqlCa.getSqlCode();
    }

    private class CursStruct {
        String fiId;
        String ident;
        String statusIdent;
        String instrumentenstatus;
        String fiSubTyp;
        String fiSubArt;
        String fiKatSchema;
        Integer iFiKatSchema; /* Indikatorvariable */

        CursStructMapper getMapper() {
            return new CursStructMapper();
        }

        public class CursStructMapper implements RowMapper<CursStruct> {
            public CursStruct mapRow(ResultSet rs, int rowNum) throws SQLException {
                CursStruct dbObject = new CursStruct();
                int i = 1;
                ResultSetMetaData metaData = rs.getMetaData();
                int count = metaData.getColumnCount();
                if (i < count && metaData.getColumnName(i).equals("fiId")) {
                    dbObject.fiId = rs.getString("fiId");
                    i++;
                }
                if (i < count && metaData.getColumnName(i).equals("ident")) {
                    dbObject.ident = rs.getString("ident");
                    i++;
                }
                if (i < count && metaData.getColumnName(i).equals("statusIdent")) {
                    dbObject.statusIdent = rs.getString("statusIdent");
                    i++;
                }
                if (i < count && metaData.getColumnName(i).equals("instrumentenstatus")) {
                    dbObject.instrumentenstatus = rs.getString("instrumentenstatus");
                    i++;
                }
                if (i < count && metaData.getColumnName(i).equals("fiSubTyp")) {
                    dbObject.fiSubTyp = rs.getString("fiSubTyp");
                    i++;
                }
                if (i < count && metaData.getColumnName(i).equals("fiSubArt")) {
                    dbObject.fiSubArt = rs.getString("fiSubArt");
                    i++;
                }
                if (i < count && metaData.getColumnName(i).equals("fiKatSchema")) {
                    dbObject.fiKatSchema = rs.getString("fiKatSchema");
                    i++;
                }
                if (i < count && metaData.getColumnName(i).equals("iFiKatSchema")) {
                    dbObject.iFiKatSchema = rs.getInt("iFiKatSchema");
                }
                return dbObject;
            }
        }
    }
}
